/********************************/
/*** HOW TO FIT AN MCMC MODEL ***/
/*** USING NHANES DATA AND    ***/
/*** BRR WEIGHTS.             ***/
/*** 10 STEP PROCESS          ***/
/********************************/

/*** This example uses prepared data from the NHANES 2011-12 cycle.
/*** The BRR weights have been prepared and included with this data.
/*** Example programs to prepare the data and create the BRR weights
/*** are available on the web site. See step1.create.data1112.sas.
/***  
/*** The following files are required to use this example:
/*** NHANES data set  - FORHEI  
/*** SAS macro		  - explore episodic n macro.sas 
/*** SAS macro        - fitdistrib macro.sas         
/*** SAS macro        - std_cov_boxcox24hr_conday_minamt_macro_v1.01.sas
/*** SAS macro        - multivar_mcmc_macro_v1.01.sas
/*** SAS macro        - multivar_distrib_macro_v1.0.sas
/*** SAS macro        - runscorese macro.sas;
/*** SAS macro        - hei2015.score.macro.sas;

/*** Assign libnames for the data set and macros using macro variables***/
/*** Note: if using the UNIX system, you will nee to replace backslashes with forward slashes ***/

options nonotes;

%global  home macros titles liblabel;

%let home     = H:\Usual Intake\HEI\final code\web example\HEI wrapper;
%let liblabel = &home\dataout\; **make sure you have created a subfolder within your home folder called 'dataout' (or whatever name you choose) for all output;

libname data "&home\data"; **make sure you have put the dataset in a subfolder of the home folder called 'data' (or whatever name you choose);

/*** Include all macros ***/
/*** This code assumes all macros are stored in a subfolder of home called 'macros' ***/

%include "&home\macros\std_cov_boxcox24hr_conday_minamt_macro_v1.01.sas"; 
%include "&home\macros\multivar_mcmc_macro_v1.01.sas";
%include "&home\macros\multivar_distrib_macro_v1.0.sas";
%include "&home\macros\hei2015.score.macro.sas";

%include "&home\macros\explore episodic n macro.sas";
%include "&home\macros\fitdistrib macro.sas";
%include "&home\macros\runscorese macro.sas";


%let titles = 0;

title%eval(&titles+1) "How to Fit an MCMC Model & Calculate HEI Scores Using NHANES Data and BRR Weights" ;
 run;


/***************************************************************/
/*** STEP 1: DATA PREPARATION                                ***/
/*** A dataset with all food groups computed was prepared    ***/
/*** in the example program step1.create.data1112.sas.       ***/
/*** The food groups needed to run MCMC are:                 ***/
/***  FOODS:                                                 ***/
/***   * Whole Fruit                                         ***/
/***   * Fruit Juice                                         ***/
/***   * Dark Green Vegetables                               ***/
/***   * Beans and Peas (legumes)                            ***/
/***   * Whole Grains                                        ***/
/***   * Refined Grains 	                                 ***/
/***   * Dairy                                               ***/
/***   * Seafood, Soy, and Nuts and seeds                    ***/
/***   * Meat, Poultry and Eggs                              ***/
/***   * Added Sugars  	                                     ***/
/***  NUTRIENTS:                                             ***/
/***   * Fatty Acids (sum of MUFA and PUFA)                  ***/
/***   * Saturated Fats                                      ***/
/***   * Sodium  	                                         ***/
/***   * Total Energy                                        ***/
/***************************************************************/


/***************************************************************/
/*** STEP 2: DETERMINE STRATUM                               ***/
/*** This is an optional step when fitting an MCMC model.    ***/
/*** It was done for NHANES 11-12 to allow for different     ***/
/*** variance components by strata. To use this code without ***/
/*** strata, we suggest creating a dummy variable called     ***/
/*** STRATUM and setting it equal to 1 for all observations. ***/
/***                                                         ***/
/*** Note: when using complex survey data, strata should be  ***/
/*** selected according to the sampling strata.              ***/
/***************************************************************/

******************************************************************
*** In this example, we used children (M & F), male adults and ***
*** female adults, using NHANES sample strata to determine age ***
*** cutoffs                                                    ***
******************************************************************;

DATA HEI;
SET DATA.FORHEI;
IF 2 <= RIDAGEYR <= 11 then STRATUM=1;
IF RIDAGEYR >= 12 and MALE=1 then STRATUM=2;
IF RIDAGEYR >= 12 and MALE=0 then STRATUM=3;
RUN;



*****************************************
*** create a dataset for each stratum ***
*****************************************;
data HEI1 HEI2 HEI3;
set HEI;
if stratum=1 then output HEI1;
if stratum=2 then output HEI2;
if stratum=3 then output HEI3;
run;



title%eval(&titles+2) "STEP 3: EXPLORE DATA CHARACTERISTICS  " ; run;

%put Starting STEP 3: EXPLORE DATA CHARACTERISTICS ;

/****************************************************************/
/*** STEP 3: EXPLORE DATA CHARACTERISTICS                     ***/
/*** This will run by stratum (use variable named STRATUM).   ***/
/*** The MCMC approach requires "uncoupling" the HEI dietary  ***/
/*** components in a different way than other methods.        ***/
/*** Therefore, some food groupings are specific to the MCMC  ***/
/*** macro only, and some that are used in the other methods  ***/
/*** of obtaining HEI scores are not used.  These variables   ***/
/*** are described as "algorithm" in the code below, to       ***/
/*** specify that they are part of the scoring algoritm       ***/
/*** fit for the standard algorithm (usual), or mcmc.         ***/
/*** the "algorithm only" components to run MCMC we include   ***/
/*** them in the example for completeness.                    ***/
/*** Note that to run the explore macro it is not necessary   ***/
/*** to distinguish "algorithm" from "mcmc" components.       ***/
/*** This is done in the SAS code for explanation only.       ***/
/*** The external SAS macro file explore episodic n macro.sas ***/ 
/*** contains the macro explore, which is called next.        ***/
/****************************************************************/

/**************************************************************************************/
/***                           ---- Parameter Names ----                            ***/ 
/***                                                                                ***/
/*** data           = the name of the data set                                      ***/
/*** id             = the name of the variable with the unique personal identifier  ***/
/***                                                                                ***/
/***    ---- The following paramters identify the food groups to be used ---        ***/
/*** wholefruit     =  Intact or cut fruit in cup equivalents                       ***/
/*** wholegrains    =  Whole grains in oz. equivalents                              ***/
/*** refinedgrains  =  Refined grains in oz. equivalents                            ***/
/*** dairy          =  Total dairy in cup equivalents                               ***/
/*** sugars         =  Added sugars in teaspoon equivalents                         ***/
/*** satfat         =  Saturated fatty acids in grams                               ***/
/*** sodium         =  Sodium in milligrams                                         ***/
/*** kcal           =  Total kcals (energy)                                         ***/
/*** mufapufa       =  Monounsaturated+polyunsaturated fatty acids in grams         ***/
/*** totalfruit     =  Total fruit in cup equivalents                               ***/
/*** beansandgreens =  Beans, peas and dark green vegetables in cup equivalents     ***/
/*** vegandlegumes  =  All vegetables plus legumes in cup equivalents               ***/
/*** seafoodplant   =  Seafood, fish and plant protein (includes legumes) in oz. eq ***/
/*** allprotein     =  Protein in oz. equivalents                                   ***/
/*** fruitjuice     =  Fruit juice in cup equivalents                               ***/
/*** nondarkgr      =  Non-dark green vegetables in cup equivalents                 ***/
/*** darkgr         =  Dark green vegetables in cup equivalents                     ***/
/*** legumes        =  Legumes in cup equivalents                                   ***/
/*** seasoynut      =  Seafood, fish, soy protein, nuts & seeds in oz. equivalents  ***/
/*** mpeggs         =  Meat, Poultry and Eggs in oz. equivalents                    ***/
/**************************************************************************************/ 

*ods noresults; **this keeps the pdf from opening, but it will still be saved in the liblabel folder;

%explore(HEI,seqn,
/*mcmc and algorithm n=9 */
wholefruit=FWHOLEFRT, wholegrains=G_WHOLE, refinedgrains=G_REFINED, dairy=D_TOTAL, sugars=ADD_SUGARS, satfat=TSFAT, sodium=TSODI, kcal=TKCAL, mufapufa=MONOPOLY, 
/*algorithm only n=5 */
totalfruit=F_TOTAL, beansandgreens=VDRKGRLEG, vegandlegumes=VTOTALLEG, seafoodplant=PFSEAPLANTLEG, allprotein=PFALLPROTLEG, 
/*mcmc only n=6 */
fruitjuice=F_JUICE, nondarkgr=VNONDRKGR, darkgr=V_DRKGR, legumes=V_LEGUMES, seasoynut=PFSEASOYNUT, mpeggs=PFMPEGGS)
  ;

/**************************************************************************/
/*** We have developed a SAS wrapper macro called fitdistrib.sas        ***/ 
/*** that fits 5 steps to be used when interest is on obtaining         ***/
/*** distributions                                                      ***/
/***                                                                    ***/
/*** The five steps are:                                                ***/
/***                                                                    ***/
/*** STEP 4: FIND MINIMUM VALUES                                        ***/
/*** Find the minimum value for all components identified as daily by   ***/
/*** stratum                                                            ***/
/***                                                                    ***/
/*** STEP 5: FIND LAMBDAS                                               ***/
/*** This step identifies the best Box Cox transformation               ***/
/*** for each dietary component by stratum                              ***/
/***                                                                    ***/
/*** STEP 6: TRANSFORM DATA TO STANDARD NORMAL                          ***/
/*** This step runs the std_cov_boxcox24hr_conday_minamt_macro_v1.0.sas ***/
/*** macro to create the input data set.                                ***/
/***                                                                    ***/
/*** STEP 7: RUN MCMC MODEL FITTING                                     ***/
/*** This is the most time consuming step. For NHANES 11/12 it takes    ***/
/*** about 1 hour to run 1 replicate for 1 stratum using 5000 MCMC      ***/
/*** iterations and 3000 burnins                                        ***/
/*** Note: Because this is so time consuming we recommend running the   ***/
/*** base replicate (0) first, then the other replicates                ***/
/*** The base replicate can be run first by setting repliclast=0        ***/
/***                                                                    ***/
/*** STEP 8: RUN DISTRIBUTION MACRO                                     ***/
/*** This macro will create the multivariate distributions of usual     ***/
/*** intake for all of the dietary components. This information will    ***/
/*** will be saved for each replicate and stratum and used in the HEI   ***/
/*** scoring macros.                                                    ***/
/**************************************************************************/

*************************************************************************
** In this example there are 3 strata. The covariates in stratum 1    ***
** (children aged 2-12) are not the same as those used in strata 2    ***
** and 3 (males >12 and females >12).                                 ***
*************************************************************************; 

ods select all;
option nonotes;

/**********************************************************************************/
/***           the parameters in the fitdistrib macro are used as follows.      ***/
/***                                                                            ***/
/*** data             =   the name of the data set prepared in steps 1-3.       ***/
/***                      Used in fitdistrib(getlambda,getminu),                ***/
/***                      std_cov_boxcox24hr_conday_minamt                      ***/
/***                                                                            ***/
/*** stratum          =   the variable that defines the strata level.           ***/
/***                      values shoud be numbers. If no strata are selected    ***/
/***                       =the value must be 1.                                ***/
/***                      Used in fitdistrib(getlambda,getminu),                ***/
/***                      std_cov_boxcox24hr_conday_minamt, multivar_distrib    ***/
/***                                                                            ***/
/*** id               =   A variable that defines a unique subject identifier   ***/
/***                      and is passed to the multivar_mcmc parameter subject. ***/
/***                      Used in  multivar_mcmc                                ***/
/***                                                                            ***/
/*** repeat           =   Specifies a variable that indexes repeated            ***/
/***                      observations for each subject.                        ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** baseweight       =   The name of the original sample weight variable       ***/
/***                      ending in _0. (e.g. Repwt_0)                          ***/
/***                      Used in fitdistrib(getlambda)                         ***/
/***                                                                            ***/
/*** weightvarroot    =   The root name for the base and other weights used for ***/
/***                      SE such as BRR weights (e.g. Repwt_). This is passed  ***/
/***                      to the multivar_mcmc macro parameter weight_var and   ***/
/***                      used in conjunction witht the replicate number        ***/
/***                      More details are given in the SAS macro code for      ***/
/***                      multivar_mcmc under weight_var.                       ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** episodic         =   The names of the dietary components consumed          ***/
/***                      episodically                                          ***/
/***                      Used in fitdistrib, std_cov_boxcox24hr_conday_minamt  ***/
/***                                                                            ***/
/*** daily            =   The names of the dietary components that are defined  ***/
/***                      daily                                                 ***/
/***                      Used in fitdistrib, std_cov_boxcox24hr_conday_minamt  ***/
/***                                                                            ***/
/*** covlisti         =   The names of the covariates that are indicator        ***/
/***                      (dummy) variables                                     ***/
/***                      Used in fitdistrib and also passed to several         ***/
/***                      paramters in the multivar_mcmc and multivar_distrib   ***/
/***                      macros.                                               ***/
/***                      More details are given in the SAS macro code for      ***/
/***                      multivar_mcmc and mulitvar_distrib.                   ***/
/***                      Used in fitdistrib(getlamba,getminu), multivar_mcmc,  ***/
/***                      mulitvar_distrib                                      ***/
/***                                                                            ***/
/*** covlistc         =   The names of the covariates that are continuous       ***/
/***                      variables                                             ***/
/***                      Used in fitdistrib and also passed to several         ***/
/***                      paramters in the multivar_mcmc and multivar_distrib   ***/
/***                      macros.                                               ***/
/***                      More details are given in the SAS macro code for      ***/
/***                      multivar_mcmc and mulitvar_distrib.                   ***/
/***                      Used in fitdistrib(getlamba,getminu), multivar_mcmc,  ***/
/***                      mulitvar_distrib                                      ***/
/***                                                                            ***/
/*** sequence         =   An indicator variable that indicates whether the      ***/
/***                      recall is from day 1 or not. It is coded 1 if day=2   ***/
/***                      and coded 0 if day=1.                                 ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** weekend          =   An indicator variable that is set to 1 for weekends   ***/
/***                      and 0 for weekdays.                                   ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** outstd           =   A user supplied name for a data set output from the   ***/
/***                      std_cov_boxcox24hr_conday_minamt macro                ***/
/***                      and passed to the data parameter of the               ***/
/***                      multivar_mcmc macro that contains the standardized    ***/
/***                      [N(0,1)] variables.                                   ***/
/***                      Saved in std_cov_boxcox24hr_conday_minamt.            ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** tranout          =   A user supplied name for a dataset output from the    ***/ 
/***                      std_cov_boxcox24hr_conday_minamt macro                ***/
/***                      and passed to the optional_iml_store_data parameter   ***/
/***                      of the multivar_mcmc macro that contains the lamdas   ***/
/***                      for the Box-Cox transformation and backtransformation.***/
/***                      Saved in std_cov_boxcox24hr_conday_minamt.            ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** replicfirst      =   A number for the first replicate fit in the           ***/
/***                      multivar_mcmc and multivar_distrib macros using       ***/
/***                      weights (in a loop from replicfirst to repliclast)    ***/
/***                      Used in fitdistrib                                    ***/
/***                                                                            ***/
/*** repliclast       =   A number for the last replicate fit in the            ***/
/***                      multivar_mcmc and multivar_distrib macros using       ***/
/***                      weights (in a loop from replicfirst to repliclast)    ***/
/***                      Used in  fitdistrib                                   ***/
/***                                                                            ***/
/*** n_mcmc_iterations=   The total number of iterations from the MCMC          ***/
/***                      algorithm including the burn-in                       ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** n_burn_iterations=   Specifies the burn-in value for the MCMC algorithm    ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** thin             =   Specifies the thinning value for the MCMC algorithm   ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** folder           =   File reference to define a folder for the             ***/
/***                      multivar_mcmc output.                                 ***/
/***                      Used in fitdistrib                                    ***/
/***                                                                            ***/
/*** outpath          =   Defines the libname for output                        ***/
/***                      used in fitdistrib                                    ***/
/***                                                                            ***/
/*** gen_inverse      =   If this is defined as "n" or "N" then the SAS IML     ***/
/***                      function "inv" is used to compute the inverse of      ***/
/***                      matrices in the MCMC algorithm. It may save           ***/
/***                      time but the user may encounter an error that the     ***/
/***                      matrix should be non-singular. By default the "ginv"  ***/
/***                      is used to compute the Moore-Penrose                  ***/
/***                      generalized inverse of matrices needed in the         ***/
/***                      MCMC algorithm.                                       ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** print            =   Determines whether macro results are printed          ***/
/***                      Used in std_cov_boxcox24hr_conday_minamt,             ***/
/***                      multivar_mcmc, multivar_distrib                       ***/
/***                                                                            ***/
/*** ntitles          =   Specifies the number of title lines to be reserved    ***/
/***                      for the user's titles                                 ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** store_sd         =   Determines whether sample standard deviations are     ***/
/***                      printed and stored in the IML storage catalog.        ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** notes_print_mcmc =   Determines whether the notes are printed to the       ***/
/***                      SAS log                                               ***/
/***                      Used in multivar_mcmc                                 ***/
/***                                                                            ***/
/*** npseudopeople    =   The number of pseudo-people used to estimate the      ***/
/***                      distribution of intake                                ***/
/***                      Used in multivar_distrib                              ***/
/***                                                                            ***/
/*** seed             =   A starting seed used by the fitdistrib macro before   ***/
/***                      calling the multivar_mcmc and multivar_distrib macros.***/
/***                      Used in fitdistrib                                    ***/
/**********************************************************************************/

/**********************************************************************************/
/*** IMPORTANT NOTE: At this time, the fitdistrib macro assumes that sequence   ***/
/***                 and weekend adjustments are made, so please include these  ***/
/***                 when using this macro wrapper                              ***/
/**********************************************************************************/

 
*************************************
** Stratum 1 - children aged 2-12  **
*************************************;

ods noresults; **this keeps the pdf from opening, but it will still be saved in the liblabel folder;

%fitdistrib
       (data               = hei1,
		stratum            = 1,
		id                 = seqn,
		repeat             = day,
		baseweight         = RepWt_0,
		weightvarroot      = RepWt_,
		episodic           = g_whole fwholefrt f_juice pfseasoynut v_drkgr v_legumes,
		daily              = add_sugars d_total g_refined tkcal tsfat tsodi pfmpeggs vnondrkgr monopoly,
		covlisti           = male raceeth_nhbl raceeth_asian raceeth_hisp age6_11,
		covlistc           = ,
		sequence           = recallday2,
		weekend            = weekend,
		outstd             = stdcov_s1,
		tranout            = tranout1,
		replicfirst        = 0,
		repliclast         = 16,  
		n_mcmc_iterations  = 5000, 
        n_burn_iterations  = 3000, 
        thin               = 10,  
		folder             = mcmcdout1,
		outpath            = "&liblabel",
        gen_inverse        = y,
        print              = n,
        ntitles            = 3,
        store_sd           = n,
        notes_print_mcmc   = y,
		npseudopeople      = 100,
        seed               = 42941); 

*************************************
** Stratum 2 - males >12           **
*************************************;

%fitdistrib    
   (data               = hei2,
		stratum            = 2,
		id                 = seqn,
		repeat             = day,
		baseweight         = RepWt_0,
		weightvarroot      = RepWt_,
		episodic           = g_whole fwholefrt f_juice pfseasoynut v_drkgr v_legumes,
		daily              = add_sugars d_total g_refined tkcal tsfat tsodi pfmpeggs vnondrkgr monopoly,
		covlisti           = raceeth_nhbl raceeth_asian raceeth_hisp age20_39 age40_49 age50_59 age60_90,
		covlistc           = ,
		sequence           = recallday2,
		weekend            = weekend,
		outstd             = stdcov_s2,
		tranout            = tranout2,
		replicfirst        = 0,
		repliclast         = 16,
		n_mcmc_iterations  = 5000,
        n_burn_iterations  = 3000,
        thin               = 10,  
		folder             = mcmcdout2,
		outpath            = "&liblabel",
        gen_inverse        = y,
        print              = n,
        ntitles            = 3,
        store_sd           = n,
        notes_print_mcmc   = y,
		npseudopeople      = 100,
        seed               = 42941);


*************************************
** Stratum 3 - females >12         **
*************************************;

%fitdistrib    
       (data               = hei3,
		stratum            = 3,
		id                 = seqn,
		repeat             = day,
		baseweight         = RepWt_0,
		weightvarroot      = RepWt_,
		episodic           = g_whole fwholefrt f_juice pfseasoynut v_drkgr v_legumes,
		daily              = add_sugars d_total g_refined tkcal tsfat tsodi pfmpeggs vnondrkgr monopoly,
		covlisti           = raceeth_nhbl raceeth_asian raceeth_hisp age20_39 age40_49 age50_59 age60_90,
		covlistc           =,
		sequence           = recallday2,
		weekend            = weekend,
		outstd             = stdcov_s3,
		tranout            = tranout3,
		replicfirst        = 0,
		repliclast         = 16,
		n_mcmc_iterations  = 5000,
        n_burn_iterations  = 3000,
        thin               = 10,  
		folder             = mcmcdout3,
		outpath            = "&liblabel",
        gen_inverse        = y,
        print              = n,
        ntitles            = 3,
        store_sd           = n,
        notes_print_mcmc   = y,
		npseudopeople      = 100,
        seed               = 42941);

*/
/*********************************************************************************/
/* Once the calls to the macro fitdistrib are completed you will need the        */
/* data distribution data sets saved from the multivar_distrib macro called      */
/* in STEP 8 of the fitdistrib macro.                                            */
/*                                                                               */
/* The MULTIVAR_DISTRIB macro creates a SAS data set called "mc_t_distrib_out"   */
/* which is saved to disk for each stratum and each replicate using              */
/* the naming convention liblabel.distmc&stratum.&replicnum -                    */
/*  e.g. liblabel.distmc01 indicates stratum1 replicate 0 for distmc.            */
/*                                                                               */
/* the data set includes usual intake variables named:                           */
/*          mc_t1   mc_t2   ...   mc_tp                                          */
/* where p = num_epis_diet_comp + num_daily_diet_comp, and the dietary           */
/* component order (1, 2, ..., 15) is equivalent to the order used when the      */
/* multivariate measurement error model was fit using the MULTIVAR_MCMC          */
/* macro.  The "mc_t_distrib_out" data set also includes the variable:           */
/*          weight_nw_sumw                                                       */
/* created and stored by the MULTIVAR_MCMC macro. This variable is defined as:   */
/* the number of subjects, n, multiplied by the input weight value and divided   */
/* by the sum of the input weight values.                                        */
/*                                                                               */
/* The number of observations in this output data set is the product of the      */
/* "number of individuals" and the specified value for the                       */
/* "set_number_monte_carlo_rand_obs" macro parameter, which is set to 500.       */
/*                                                                               */
/*********************************************************************************/
 
************************************************************************
*** end of the multivar distrib process steps 4-8.                   ***
************************************************************************;

************************************************************************
*** After you run all replicates you can move onto scoring           ***
************************************************************************;

option nonotes;

/**************************************************************************/
/*** The SAS wrapper macro called runscorese.sas                        ***/ 
/*** scores the simulated distribution data for the HEI-2015            ***/
/*** and also calculates SEs using BRR                                  ***/
/***                                                                    ***/
/*** The two steps are:                                                 ***/
/***                                                                    ***/
/*** STEP 9: SCORE EACH REPLICATE                                       ***/
/*** Create ratios and score each one according to the relevant         ***/
/*** standard                                                           ***/
/***                                                                    ***/
/*** STEP 10: CALCULATE STANDARD ERRORS USING BRR                       ***/
/*** Use the BRR replicates to create standard errors for the mean and  ***/
/*** the percentiles of interest                                        ***/
/***                                                                    ***/
/**************************************************************************/

/*****************************************************************************************/
/***           --- The paramters are  ---                                              ***/
/*** outpath           =  Folder where the distmc variables are and the location       ***/
/***                      for the data set to be saved                                 ***/
/*** replicfirst       =  The starting number for macro calls used to score the HEI.   ***/ 
/***                      Usually 0.                                                   ***/
/*** replicfirstse     =  The starting number for the macro calls used in calculating  ***/ 
/***                      the SEs for the scores. Usually 1.                           ***/
/*** repliclast        =  The ending number for the calls to the macros. The number    ***/
/***                      of BRR weights used in the analysis.                         ***/
/*** finaldataset      =  A valid SAS name for the final data set                      ***/
/*** episodic          =  A list of episodic variable names - separated by spaces      ***/
/*** daily             =  A list of non-episodice variable names - separated by spaces ***/
/*** wholefruit        =  A valid SAS variable name to be output for wholefruit        ***/
/*** beansandgreens    =  A valid SAS variable name to be output for beansandgreens    ***/
/*** wholegrains       =  A valid SAS variable name to be output for wholegrains       ***/
/*** refinedgrains     =  A valid SAS variable name to be output for refinedgrains     ***/
/*** dairy             =  A valid SAS variable name to be output for dairy             ***/
/*** sugars            =  A valid SAS variable name to be output for sugars            ***/
/*** satfat            =  A valid SAS variable name to be output for satfat            ***/
/*** sodium            =  A valid SAS variable name to be output for sodium            ***/
/*** kcal              =  A valid SAS variable name to be output for kcal              ***/
/*** totalfruit        =  A valid SAS variable name to be output for totalfruit        ***/             
/*** vegandlegumes     =  A valid SAS variable name to be output for vegandlegumes     ***/
/*** seafoodplant      =  A valid SAS variable name to be output for seafoodplant      ***/
/*** allprotein        =  A valid SAS variable name to be output for all protein       ***/
/*** mufapufa          =  A valid SAS variable name to be output for mufapufa          ***/
/*** fruitjuice        =  A valid SAS variable name to be output for fruitjuice        ***/
/*** nondarkgr         =  A valid SAS variable name to be output for nondarkgr         ***/
/*** darkgr            =  A valid SAS variable name to be output for darkgr            ***/
/*** legumes           =  A valid SAS variable name to be output for legumes           ***/
/*** seasoynut         =  A valid SAS variable name to be output for seasoynut         ***/
/*** mpeggs            =  A valid SAS variable name to be output for mpeggs            ***/
/*****************************************************************************************/                                               
option nonotes;
ods select all;

/*****************************************************************************************/                                               
/*** The runscorese macro will produce two pdf files: hei_scores_se and hei_plots        */
/*****************************************************************************************/                                               


%runscorese(outpath=&liblabel, replicfirst=0, replicfirstse=1, repliclast=16, finaldataset=tdheistat16,
episodic=g_whole fwholefrt f_juice pfseasoynut v_drkgr v_legumes,
daily=add_sugars d_total g_refined tkcal tsfat tsodi pfmpeggs vnondrkgr monopoly,
wholefruit=fwholefrt, beansandgreens=VDRKGRLEG, wholegrains=G_WHOLE, refinedgrains=G_REFINED, 
dairy=D_TOTAL, sugars=ADD_SUGARS, satfat=tsfat, sodium=tsodi, 
kcal=TKCAL, totalfruit=F_TOTAL, vegandlegumes=VTOTALLEG, seafoodplant=PFSEAPLANTLEG, allprotein=PFALLPROTLEG, 
mufapufa=MONOPOLY, fruitjuice=F_JUICE, nondarkgr=VNONDRKGR, darkgr=V_DRKGR, legumes=V_LEGUMES, 
seasoynut=PFSEASOYNUT, mpeggs=PFMPEGGS)


option notes;
************************************************************************
*** Congratulations you have just fit an MCMC Distribution!          ***
************************************************************************;

  
